/*
 i-net software provides programming examples for illustration only, without warranty
 either expressed or implied, including, but not limited to, the implied warranties
 of merchantability and/or fitness for a particular purpose. This programming example
 assumes that you are familiar with the programming language being demonstrated and
 the tools used to create and debug procedures. i-net software support professionals
 can help explain the functionality of a particular procedure, but they will not modify
 these examples to provide added functionality or construct procedures to meet your
 specific needs.
  
 � i-net software 1998-2013

*/
using System;
using System.Collections.Generic;
using Inet.Viewer.Data;
using Inet.Viewer.WinForms;

namespace Inet.Viewer
{
    /// <summary>
    ///  Event that is fired when the CurrentPage was changed
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="page"></param>
    public delegate void PageChanged(object sender, int page);

    /// <summary>
    /// Event that is fired when the data changes was changed. This can occur after a refesh or a restart of the server.
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="timestamp"></param>
    public delegate void DataChanged(object sender, long timestamp);

    /// <summary>
    /// The mouse action mode that defines what should happen if the left mouse button was clicked within the displayed report
    /// </summary>
    public enum MouseMode
    {
        /// <summary>
        /// Click and drag causes panning of the page along with the mouse position
        /// movement.        
        /// </summary>
        Panning = 0,

        /// <summary>
        /// Click and drag causes a text selection, line by line.
        /// </summary>
        SelectText = 1,

        /// <summary>
        /// ReportView.MODE_SNAPSHOT - Click and drag causes an image snapshot within a box at release of the mouse button.
        /// </summary>
        TakeSnapshot = 2
    }

    /// <summary>
    /// This enum defines the different zoom modes
    /// </summary>
    public enum ZoomMode
    {
        /// <summary>
        /// To be able the set manual Zoom factor <see cref="Inet.Viewer.IReportView.ZoomFactor"/>.  No automatic zoom.        
        /// </summary>    
        Manual = 0,

        /// <summary>
        /// Automatically zoom in or out so that the full width of the page is visible.        
        /// </summary>
        PageWidth = 1,

        /// <summary>
        /// Automatically zoom in or out so that the full height of the page is visible.        
        /// </summary>
        PageHeight = 2,

        /// <summary>
        /// Automatically zoom in or out so that the full page is visible. This
        /// is equivalent to PageHeight when the page is a portrait view, or to PageWidth
        /// when the page is a landscape view.        
        /// </summary>
        FullPage = 3
    }

    /// <summary>
    /// ViewMode to distinguis the different Page View Modes
    /// </summary>
    public enum PageViewMode
    {
        /// <summary>
        /// Only one page is shown at a time.        
        /// </summary>
        SinglePage = 1,

        /// <summary>
        /// The report can be scrolled through in its entirety, with each page being shown above the next.        
        /// </summary>
        SingleContinuous = 2,

        /// <summary>
        /// Only two pages are shown at a time, one next to the other.        
        /// </summary>
        DoublePage = 3,

        /// <summary>
        /// The report can be scrolled through in its entirety, however showing the pages two by two.        
        /// </summary>
        DoubleContinuous = 4
    }

    /// <summary>
    /// Used to distinguish the different Loading Status
    /// </summary>
    public enum Status
    {
        /// <summary>
        /// The report view is loading for the first time, no pages have been loaded yet.         
        /// </summary>
        Initialized = 0,

        /// <summary>
        /// At least one page has been loaded and the report view is currently fetching 
        /// the other pages of the report if there are any        
        /// </summary>
        Loading = 1,

        /// <summary>
        /// The report has been successfully loaded         
        /// </summary>
        Finished = 2,

        /// <summary>
        /// The loading has been canceled by the user         
        /// </summary>
        Canceled = -1
    }

    /// <summary>
    /// The ReportView is the top-level instance for a report, it holds references to the report's
    /// NavigationView and StatusBar. It offers various navigation methods, as well as the possibility to
    /// extract components and replace them with own components.<br/>
    /// To obtain an instance of a ReportView, use the methods <seealso cref="IReportViewer.AddNewReportView(IRenderData)"/> <br/> 
    /// </summary>
    public interface IReportView
    {

        /// <summary>
        /// Displays an error box showing the Throwable's error message. If a ViewerException is thrown, the special messages and properties
        /// of the ViewerException are shown. Otherwise, only the getMessage() as well as the stack trace are shown.
        /// </summary>
        /// <param name="th"> Throwable to display in an error box. Best if this Throwable is a ViewerException.</param> 
        void ShowError(Exception th);

        /// <summary>
        /// Causes this ReportView to refresh its pages by re-fetching the report data - this causes a
        /// re-rendering of the report data. If "promptOnRefresh" is set, this will cause the prompts
        /// to be fetched again as well. </summary>
        void DataRefresh();
 
        /// <summary>
        /// Causes the page indicated by the parameter pageNum to be displayed in this ReportView. If
        /// this page does not exist in the report, calling this method will do nothing at all. Note this
        /// is an asynchronous call - the page will be displayed at some point, but it is not guaranteed
        /// that the page is actually visible as soon as this method returns. </summary>
        /// <param> Number of page (starting at 1) to be displayed in this ReportView. </param>
        /// <summary>
        /// Returns the number of the "current page" being displayed in this ReportView. If more than
        /// one PageView is visible, this returns the lowest page visible at this time. If no PageView
        /// is visible, this returns 0. </summary>
        /// <returns> Number of the first page currently visible in this ReportView, or 0 if no page is
        /// visible.
        /// </returns>         
        int CurrentPage { get; set; }


        /// <summary>
        /// Returns the total number of pages this report has. If the total number of pages is as of
        /// yet unknown (for example if the report is still being rendered), this will return
        /// TOTAL_PAGE_COUNT_UNKNOWN. </summary>
        /// <returns> Total number of pages in the report, or TOTAL_PAGE_COUNT_UNKNOWN if still unknown. </returns>     
        int PageCount { get; }

        /// <summary>
        /// Navigates one step back in the report - if this is not possible (e.g. the current page is
        /// already the first page of the report), this method does nothing.   
        /// </summary>
        void PreviousPage();

        /// <summary>
        /// Navigates one page forward in the report.  If this is not possible e.g. the current page is
        /// already the last page of the report, this method does nothing.  
        /// </summary>
        void NextPage();

        /// <summary>
        /// Navigates directly to the last page in the report. If  the current page is
        /// already the last page of the report, or if the total page count is not known
        /// as of yet (that is, the report is still being rendered), this method does nothing. Note this
        /// is an asynchronous call - the page will be displayed at some point, but it is not guaranteed
        /// that the page is actually visible as soon as this method returns.  
        /// </summary>
        void LastPage();

        /// <summary>
        /// Navigates directly to the first page of the report
        /// </summary>
        void FirstPage();


        /// <summary>
        /// Sets the type of view this ReportView is to take and causes this view to be displayed
        /// immediately.    
        /// </summary>
        /// <param> Layout type to display in this ReportView. </param>
        /// <seealso cref= "PageViewMode"> </seealso>      
        PageViewMode ViewMode { set; get; }

        /// <summary>
        /// Set and get the ReportInfo for this ReportView
        /// </summary>
        ReportInfo ReportInfo { get; set; }

        /// <summary>
        /// Returns the parent ReportViewer for this component. </summary>
        /// <returns> the parent ReportViewer for this component  </returns>      
        IReportViewer ReportViewer { get; set; }

        /// <summary>
        /// Sets the zoom factor to this value. Only values greater than 0 are allowed. This value is
        /// not a percentage value, that is, 2.0 is 2x view, 0.25 is 1/4 view, etc.<br/>
        /// If the value is less than MIN_ZOOM_FACTOR, this has the same effect as calling setZoomFactor(MIN_ZOOM_FACTOR).<br/>
        /// If the value is greater than MAX_ZOOM_FACTOR, this has the same effect as calling setZoomFactor(MAX_ZOOM_FACTOR).<br/>
        /// If you want to zoom relative to the page's size, you have to set the <seealso cref="ZoomMode"/> accordingly.
        /// This Zoom Factor is used for the Manul Zoom Type, for the others this Property is ignored!
        /// </summary>
        /// <param> Factor to zoom with: 1.0 for 100%), 2.0 for 200%, etc. </param>
        /// <seealso cref= "ZoomMin"> </seealso>
        /// <seealso cref= "ZoomMax"> </seealso>
        /// <seealso cref= "ZoomMode"></seealso>
        float ZoomFactor { set; get; }

        /// <summary>        
        /// Defines the Minimum zoom level
        /// </summary>
        float ZoomMin { set; get; }

        /// <summary>        
        /// Defines the Maximum zoom level
        /// </summary>
        float ZoomMax { set; get; }

        /// <summary>
        /// Sets the type of zoom to use in the report view when the ReportView is displayed anew (for
        /// example after resizing the display). Possible modes are manual zoom, a zoom to page with, zoom to page
        /// height, or zoom to show full page.
        /// </summary>
        /// <param> Zoom type to use when displaying the ReportView. </param>
        /// <seealso cref= "ZoomMode">The different zoom modes </seealso> 
        ZoomMode ZoomMode { set; get; }

        /// <summary>
        /// Sets the mode for mouse actions in this ReportView, such as clicking and dragging.        
        /// <see cref="MouseMode">Enum with the different Modes</see>
        /// <param> Mouse mode to be used in the ReportView. </param>
        /// </summary>
        MouseMode MouseActionMode { set; get; }

        /// <summary>
        /// Returns the ReportData object belonging to this ReportView. The ReportData object is the source of
        /// report data for this ReportView. If this is null, the ReportView has no connection to a ReportData
        /// source and should be considered obsolete. </summary>
        /// <returns> The ReportData object belonging to this ReportView</returns> 
        IRenderData ReportData { get; set; }

        /// <summary>
        /// Returns the title of this report if it is known. If not known, the title returned will be null. </summary>
        /// <returns> The report title of this report if known. Will return null if not known </returns>
        /// <seealso cref= "IRenderData.ReportTitle"/> 
        string ReportTitle { get; set; }

        /// <summary>
        /// Returns whether this report view has more pages than the server page limit for a report allows for (causing the later pages to be truncated
        /// from the report). </summary>
        /// <returns> whether this report view has more pages than the server page limit</returns>
        bool PageLimitExceeded { get; }

        /// <summary>
        /// Returns whether this report was suppressed because it has no rows. </summary>
        /// <returns> true, if no rows.</returns>  
        bool ReportSuppressed { get; }

        /// <summary>
        /// Determines if this ReportView can be closed via the User Interface, by for example clicking on the close button        
        /// </summary>
        bool IsCloseable { get; set; }

        /// <summary>
        /// event for when the Page number changed
        /// </summary>
        event PageChanged PageChanged;

        /// <summary>
        /// Event when the Zoom changed
        /// </summary>
        event EventHandler ZoomChanged;

        /// <summary>
        /// Occured when data was initially received.
        /// </summary>
        event DataChanged DataInited;

        /// <summary>
        /// An event that is trigged when the data was updated, E.g. through the Refresh method
        /// </summary>        
        event DataChanged DataChanged;

        /// <summary>
        /// Exports the report connected to this report view into the given file.<BR/>
        /// Export formats creating multiple files, e.g. HTML and SVG, creating a sub directory
        /// with the name of the first file. All files will be saved in that directory expecting the first file.<BR/> 
        /// This method uses the default values for all parameters. 
        /// </summary>
        /// <remarks>
        /// To export a report into a file without preview in the .Net Viewer we recommend
        /// to use the Engine directly instead of using the Viewer API because the .Net Viewer
        /// has to request the first report page in the .Net Viewer format before it can
        /// request the report in the specified export format.
        /// </remarks>
        /// <param name="format"> The export format in that the report will be exported. </param>
        /// <param name="file">   Name of the file in that the report will be exported. </param>
        /// <returns> ExportProgress of running export </returns>
        /// <exception cref="ViewerException"> If exporting has been disabled for this report or the requested export format is not available for this report
        /// </exception>
        /// <see cref=" Export(Dictionary{string, string})">With this export method a set of paramters can be set </see>
        /// <seealso cref= "ExportProgress"/>   
        /// <seealso cref= "ExportFormat"/>         
        Progress Export(ExportFormat format, string file);

        /// <summary>
        /// Exports the report connected to this report view into the given file.<BR/>
        /// This method use the default value for all not specified paramters. For a complete list of report URL properties see the documentation.<br/>        
        /// <a href ="https://docs.inetsoftware.de/reporting/help/report-url-parameters"> URL Parameters</a>
        /// </summary>
        /// <remarks>
        /// To export a report into a file without preview in the .Net Viewer we recommend
        /// to use the Engine directly instead of using the Viewer API because the .Net Viewer
        /// has to request the first report page in the .Net Viewer format before it can
        /// request the report in the specified export format.
        /// </remarks>       
        /// <param name="exportParameter"> Properties for the export, including format, file and other properties if they are available for the selected format.</param>
        /// <returns> ExportProgress of running export</returns>
        /// <exception cref="ViewerException"> If exporting has been disabled for this report or the requested export format is not available for this report</exception>
        /// <seealso cref="Export(ExportFormat, string)">Simple method for export, just put in the format and the filename </seealso>
        /// <seealso cref= "ExportProgress"/>         
        Progress Export(Dictionary<string, string> exportParameter);

        /// <summary>
        /// Shows a print dialog and prints the report asynchronously.
        /// </summary>
        void Print();

        /// <summary>
        /// Highlights a number of texts in the shown report.
        /// </summary>
        /// <param name="searchChunk">an array of instances saving the text block locations and substring ranges of the texts to hightlight</param>
        void Highlight(SearchChunk[] searchChunk);

        /// <summary>
        /// Opens the export dialog for this report view.
        /// </summary>
        void OpenExportDialog();

        /// <summary>
        /// Clears any hightlighted texts.
        /// </summary>
        void ClearSelection();

        /// <summary>
        /// Focuses this control.
        /// </summary>
        void Focus();
    }
}